<?php
/**
 * Inventaire d'images
 * Auteur : Olivier Delhaye (Université Aristote de Thessalonique - Université de Chypre)
 * © 2025 Olivier Delhaye
 * Licence : GPL-3.0-or-later
 * Ce script s’appuie sur les bibliothèques de Moodle (GPL v3). Pour rester cohérent,
 * nous publions ce code sous GPL v3 (ou ultérieure).
 */
@ini_set('display_errors', isset($_GET['debug']) ? '1' : '0');
@error_reporting(isset($_GET['debug']) ? (E_ALL|E_STRICT) : 0);
define('NO_OUTPUT_BUFFERING', true);

require(__DIR__.'/config.php');
if (!empty($_GET['debug'])) { $CFG->debug = E_ALL|E_STRICT; $CFG->debugdisplay = 1; }
require_login();
require_once($CFG->libdir.'/filelib.php');

$context = context_system::instance();
require_capability('moodle/site:config', $context); // admin only

$action = optional_param('action', 'view', PARAM_ALPHA);
$tab    = optional_param('tab', 'files', PARAM_ALPHA);
$perpage= max(10, min(500, optional_param('perpage', 100, PARAM_INT)));
$page   = max(0, optional_param('page', 0, PARAM_INT));
$componentf = optional_param('component', '', PARAM_RAW_TRIMMED);
$fileareaf  = optional_param('filearea', '', PARAM_RAW_TRIMMED);
$search     = optional_param('search', '', PARAM_RAW_TRIMMED);
$download   = optional_param('download', '', PARAM_ALPHA);
$columnsraw = optional_param('columns', '', PARAM_RAW_TRIMMED);
$columns = $columnsraw !== '' ? $columnsraw : 'page.content,label.intro,book_chapters.content,forum_posts.message,resource.intro';

if ($action === 'thumb') {
    // Thumbnail / original proxy — never fatal if GD missing
    require_sesskey();
    $id = required_param('id', PARAM_INT);
    $w  = optional_param('w', 0, PARAM_INT);
    $h  = optional_param('h', 0, PARAM_INT);
    $rec = $DB->get_record('files', ['id'=>$id], '*', MUST_EXIST);
    if ($rec->filename === '.' || (int)$rec->filesize === 0 || strpos($rec->mimetype,'image/') !== 0) {
        header('Content-Type: image/svg+xml');
        echo '<svg xmlns="http://www.w3.org/2000/svg" width="120" height="80"><rect width="100%" height="100%" fill="#eee"/><text x="50%" y="50%" dominant-baseline="middle" text-anchor="middle" font-size="12" fill="#888">no image</text></svg>';
        exit;
    }
    $fs = get_file_storage();
    $file = $fs->get_file($rec->contextid, $rec->component, $rec->filearea, $rec->itemid, $rec->filepath, $rec->filename);
    if (!$file) { print_error('filenotfound'); }
    if ($w > 0 && $h > 0 && function_exists('imagecreatefromstring')) {
        $data = $file->get_content();
        if ($src = @imagecreatefromstring($data)) {
            $sw = imagesx($src); $sh = imagesy($src);
            $ratio = min($w/$sw, $h/$sh);
            $tw = max(1, (int)floor($sw * $ratio));
            $th = max(1, (int)floor($sh * $ratio));
            $dst = imagecreatetruecolor($tw, $th);
            imagealphablending($dst, false); imagesavealpha($dst, true);
            imagecopyresampled($dst, $src, 0,0,0,0, $tw,$th, $sw,$sh);
            header('Content-Type: image/png');
            header('Cache-Control: max-age=600, public');
            imagepng($dst);
            imagedestroy($dst); imagedestroy($src);
            exit;
        }
    }
    // Fallback: stream original
    send_stored_file($file, 600, 0, false); exit;
}

define('IR_VERSION', 'v8');
define('IR_AUTHORS', 'Olivier Delhaye');
define('IR_COPYRIGHT', '© 2025 Olivier Delhaye');
define('IR_LICENSE', 'GPL-3.0-or-later');


// ---------- Helpers ----------
function ir_pager_form($actionurl, $page, $pages, array $hidden = []) {
    $page = max(0, (int)$page);
    $pages = max(1, (int)$pages);
    $prev = max(0, $page - 1);
    $next = min($pages - 1, $page + 1);

    $prevurl = clone $actionurl; $prevurl->param('page', $prev);
    $nexturl = clone $actionurl; $nexturl->param('page', $next);

    echo '<div class="pager">';
    echo '<a class="btnlink" href="'.s($prevurl->out(false)).'">&laquo; Précédent</a>';

    echo '<form method="get" action="'.s($actionurl->out(false)).'" class="pagerform">';
    foreach ($hidden as $k => $v) {
        echo '<input type="hidden" name="'.s($k).'" value="'.s($v).'">';
    }
    echo '<label>Page ';
    echo '<select name="page">';
    for ($i = 0; $i < $pages; $i++) {
        $sel = $i == $page ? ' selected' : '';
        echo '<option value="'.$i.'"'.$sel.'>'.($i + 1).' / '.$pages.'</option>';
    }
    echo '</select> ';
    echo '<button type="submit">Aller</button>';
    echo '</label>';
    echo '</form>';

    echo '<a class="btnlink" href="'.s($nexturl->out(false)).'">Suivant &raquo;</a>';
    echo '</div>';
}

function ir_compact_css() {
    echo '<style>
    body{font:14px/1.4 system-ui,Segoe UI,Roboto,Arial}
    table{border-collapse:collapse;width:100%}
    th,td{border:1px solid #ddd;padding:6px}
    th{background:#f6f6f6;text-align:left}
    img{vertical-align:middle}
    .controls{margin:10px 0}
    .pager{display:flex;gap:10px;align-items:center;flex-wrap:wrap;margin:10px 0}
    .pagerform select{font-size:13px;padding:2px 6px}
    .pagerform button{font-size:13px;padding:2px 8px}
    .btnlink{padding:2px 6px;border:1px solid #ccc;border-radius:6px;text-decoration:none;background:#fafafa}
    .muted{color:#666;font-size:12px}
    </style>';
}

function ir_where(&$params,$componentf,$fileareaf,$search) {
    global $DB;
    $params=[];
    $w=["f.filename <> '.'","f.filesize > 0",$DB->sql_like('f.mimetype',':img',false,false)];
    $params['img']='image/%';
    if ($componentf!==''){ $w[]='f.component = :component'; $params['component']=$componentf; }
    if ($fileareaf!==''){ $w[]='f.filearea = :filearea'; $params['filearea']=$fileareaf; }
    if ($search!==''){
        // Distinct placeholders for LIKEs
        $like1 = $DB->sql_like('f.filename',':s1',false,false);
        $like2 = $DB->sql_like('f.source',  ':s2',false,false);
        $w[]="($like1 OR $like2)";
        $params['s1']="%$search%";
        $params['s2']="%$search%";
    }
    return 'WHERE '.implode(' AND ', $w);
}

// Cache for module id by name
function ir_modid($name){
    static $cache = [];
    global $DB;
    if (isset($cache[$name])) return $cache[$name];
    $cache[$name] = (int)$DB->get_field('modules','id', ['name'=>$name]);
    return $cache[$name];
}

// Best-effort: resolve cmid/modname from a table+row id for HTML scan
function ir_resolve_cmid_for($table, $rowid){
    global $DB;
    $rowid = (int)$rowid;
    try {
        switch ($table) {
            case 'page':
            case 'label':
            case 'resource':
                $modname = $table; // same names
                $cmid = $DB->get_field('course_modules','id', ['instance'=>$rowid, 'module'=>ir_modid($modname)]);
                if ($cmid) return ['cmid'=>(int)$cmid,'modname'=>$modname];
                break;
            case 'book_chapters':
                // book_chapters -> book -> cm(module book, instance=book.id)
                $sql = "SELECT cm.id
                          FROM {book_chapters} bc
                          JOIN {book} b ON b.id = bc.bookid
                          JOIN {course_modules} cm ON cm.instance = b.id AND cm.module = :mod
                         WHERE bc.id = :id";
                $cmid = $DB->get_field_sql($sql, ['mod'=>ir_modid('book'),'id'=>$rowid]);
                if ($cmid) return ['cmid'=>(int)$cmid,'modname'=>'book'];
                break;
            case 'forum_posts':
                // forum_posts -> forum_discussions -> forum -> cm(module forum, instance=forum.id)
                $sql = "SELECT cm.id
                          FROM {forum_posts} p
                          JOIN {forum_discussions} d ON d.id = p.discussion
                          JOIN {forum} f ON f.id = d.forum
                          JOIN {course_modules} cm ON cm.instance = f.id AND cm.module = :mod
                         WHERE p.id = :id";
                $cmid = $DB->get_field_sql($sql, ['mod'=>ir_modid('forum'),'id'=>$rowid]);
                if ($cmid) return ['cmid'=>(int)$cmid,'modname'=>'forum'];
                break;
        }
    } catch (\Throwable $e) {
        // ignore resolver failures
    }
    return ['cmid'=>null,'modname'=>null];
}

// ---------- HTML Shell ----------
echo '<!doctype html><html><head><meta charset="utf-8"><title>Inventaire images </title>';
echo '<meta name="author" content="'.s(IR_AUTHORS).'">';

ir_compact_css();
echo '</head><body>';
echo '<h2>Inventaire & nettoyage d\'images</h2>';
echo '<div class="controls muted">Page admin uniquement. '
    .(!empty($_GET['debug'])?'Debug ON':'Ajouter ?debug=1 à l’URL pour voir les erreurs PHP').'</div>';
				echo '<div class="controls muted">'
   . IR_COPYRIGHT.' | '.IR_LICENSE.'</div>';


$base = new moodle_url('/inventaire-images.php');
$tabs = ['files'=>'Images stockées','html'=>'Images intégrées & externes','dupes'=>'Doublons exacts'];
echo '<div class="controls">';
foreach ($tabs as $key=>$label) {
    $u = new moodle_url($base, ['tab'=>$key,'perpage'=>$perpage,'page'=>0,'component'=>$componentf,'filearea'=>$fileareaf,'search'=>$search,'columns'=>$columns]);
    $style = $tab===$key?'font-weight:bold;text-decoration:underline':'';
    echo '<a class="btnlink" style="'.$style.'" href="'.s($u->out(false)).'">'.s($label).'</a> ';
}
echo '</div>';

// ---------- Tabs ----------
if ($tab==='files'){
    // filters
    echo '<form class="controls" method="get" action="'.s($base->out(false)).'">';
    echo '<input type="hidden" name="tab" value="files">';
    echo '<label>Composant <input type="text" name="component" value="'.s($componentf).'"></label> ';
    echo '<label>Zone <input type="text" name="filearea" value="'.s($fileareaf).'"></label> ';
    echo '<label>Nom/Source <input type="text" name="search" value="'.s($search).'"></label> ';
    echo '<label>Par page <input type="number" name="perpage" value="'.(int)$perpage.'" min="10" max="500"></label> ';
    echo '<button type="submit">Filtrer</button>';
    echo '</form>';

    // query & counts (JOIN context + course + cm + modules)
    $where = ir_where($params,$componentf,$fileareaf,$search);
    $count = $DB->count_records_sql("SELECT COUNT(1)
                                       FROM {files} f
                                       JOIN {context} ctx ON ctx.id = f.contextid
                                  LEFT JOIN {course_modules} cm ON (ctx.contextlevel = ".CONTEXT_MODULE." AND cm.id = ctx.instanceid)
                                  LEFT JOIN {modules} m ON (m.id = cm.module)
                                  LEFT JOIN {course} c ON ((ctx.contextlevel = ".CONTEXT_COURSE." AND c.id = ctx.instanceid) OR (ctx.contextlevel = ".CONTEXT_MODULE." AND c.id = cm.course))
                                       $where", $params);

    $sql = "SELECT f.id,f.contenthash,f.contextid,f.component,f.filearea,f.itemid,f.filename,f.mimetype,f.filesize,f.timecreated,f.timemodified,
                   ctx.contextlevel, ctx.instanceid,
                   c.id AS courseid, c.fullname AS coursename,
                   cm.id AS cmid, cm.instance AS instanceid, m.name AS modname
              FROM {files} f
              JOIN {context} ctx ON ctx.id = f.contextid
         LEFT JOIN {course_modules} cm ON (ctx.contextlevel = ".CONTEXT_MODULE." AND cm.id = ctx.instanceid)
         LEFT JOIN {modules} m ON (m.id = cm.module)
         LEFT JOIN {course} c ON ((ctx.contextlevel = ".CONTEXT_COURSE." AND c.id = ctx.instanceid) OR (ctx.contextlevel = ".CONTEXT_MODULE." AND c.id = cm.course))
              $where
          ORDER BY f.timemodified DESC";

    $recs = $DB->get_records_sql($sql, $params, $page*$perpage, $perpage);
    $pages = (int)ceil($count / max(1,$perpage));

    // TOP: compact pager then CSV
    $basefiles = new moodle_url($base, ['tab'=>'files','component'=>$componentf,'filearea'=>$fileareaf,'search'=>$search,'perpage'=>$perpage]);
    ir_pager_form($basefiles, $page, $pages, ['tab'=>'files','component'=>$componentf,'filearea'=>$fileareaf,'search'=>$search,'perpage'=>$perpage]);
    $dlurl = new moodle_url($base, ['tab'=>'files','download'=>'csv','component'=>$componentf,'filearea'=>$fileareaf,'search'=>$search,'columns'=>$columns,'perpage'=>$perpage,'page'=>$page]);
    echo '<div class="controls"><a class="btnlink" href="'.s($dlurl->out(false)).'">Exporter en CSV</a></div>';

    // table
    echo '<table><tr><th>Aperçu</th><th>Nom</th><th>MIME</th><th>Taille</th><th>Composant</th><th>Zone</th><th>Course ID</th><th>CM ID</th><th>Activité</th><th>Instance</th><th>Créé</th><th>Modifié</th></tr>';
    foreach ($recs as $r) {
        $thumb = new moodle_url($base, ['action'=>'thumb','id'=>$r->id,'w'=>160,'h'=>100,'sesskey'=>sesskey()]);
        $orig  = new moodle_url($base, ['action'=>'thumb','id'=>$r->id,'sesskey'=>sesskey()]);
        $courselink = !empty($r->courseid) ? '<a href="'.s((new moodle_url('/course/view.php', ['id'=>$r->courseid]))->out(false)).'" target="_blank">'.(int)$r->courseid.'</a>' : '-';
        $cmlink = (!empty($r->cmid) && !empty($r->modname)) ? '<a href="'.s((new moodle_url('/mod/'.$r->modname.'/view.php', ['id'=>$r->cmid]))->out(false)).'" target="_blank">'.(int)$r->cmid.'</a>' : '-';
        $modlabel = !empty($r->modname) ? s($r->modname) : '-';
        $instlabel = !empty($r->instanceid) ? (int)$r->instanceid : '-';
        echo '<tr>';
        echo '<td><a href="'.s($orig->out(false)).'" target="_blank"><img src="'.s($thumb->out(false)).'" style="max-width:160px;max-height:100px;object-fit:contain"></a></td>';
        echo '<td><a href="'.s($orig->out(false)).'" target="_blank">'.s($r->filename).'</a></td>';
        echo '<td>'.s($r->mimetype).'</td>';
        echo '<td>'.display_size($r->filesize).'</td>';
        echo '<td>'.s($r->component).'</td>';
        echo '<td>'.s($r->filearea).'</td>';
        echo '<td>'.$courselink.'</td>';
        echo '<td>'.$cmlink.'</td>';
        echo '<td>'.$modlabel.'</td>';
        echo '<td>'.$instlabel.'</td>';
        echo '<td>'.userdate($r->timecreated).'</td>';
        echo '<td>'.userdate($r->timemodified).'</td>';
        echo '</tr>';
    }
    echo '</table>';

    // bottom pager (compact too)
    ir_pager_form($basefiles, $page, $pages, ['tab'=>'files','component'=>$componentf,'filearea'=>$fileareaf,'search'=>$search,'perpage'=>$perpage]);
}
elseif ($tab==='html'){
    echo '<form class="controls" method="get" action="'.s($base->out(false)).'">';
    echo '<input type="hidden" name="tab" value="html">';
    echo '<label>Colonnes (table.col, virgules) ';
    echo '<input style="width:600px" type="text" name="columns" value="'.s($columns).'"></label> ';
    echo '<label>Par page <input type="number" name="perpage" value="'.(int)$perpage.'" min="10" max="500"></label> ';
    echo '<button type="submit">Analyser</button>';
    echo '</form>';

    $cols = array_filter(array_map('trim', explode(',', (string)$columns)));
    $matches = []; $total=0; $shown=0; $offset=$page*$perpage;

    foreach ($cols as $tc) {
        if (strpos($tc,'.')===false) continue;
        list($table,$col)=explode('.',$tc,2);
        $table=preg_replace('/[^a-z0-9_]/','',$table);
        $col  =preg_replace('/[^a-z0-9_]/','',$col);
        if (!$DB->get_manager()->table_exists($table)) continue;

        // Try to detect course column
        $hascourse = array_key_exists('course', $DB->get_columns($table));
        $select = $hascourse ? "id, $col AS html, course AS courseid" : "id, $col AS html, NULL AS courseid";

        $sql="SELECT $select FROM {".$table."} WHERE $col LIKE :n1 OR $col LIKE :n2";
        $rs=$DB->get_recordset_sql($sql,['n1'=>'%<img%','n2'=>'%data:image%']);
        foreach($rs as $row){
            $html=(string)$row->html; if ($html==='') continue;

            $resolved = ir_resolve_cmid_for($table, $row->id);
            $cmid = $resolved['cmid']; $modname = $resolved['modname'];
            $actlink = ($cmid && $modname) ? '<a href="'.s((new moodle_url('/mod/'.$modname.'/view.php', ['id'=>$cmid]))->out(false)).'" target="_blank">'.(int)$cmid.'</a>' : '-';

            $where = $table.' ID '.(int)$row->id
                   . ($row->courseid ? ' (course '.$row->courseid.')' : '');

            if (preg_match_all('/<img[^>]+src\s*=\s*"(https?:[^"\s<>]+)"/i',$html,$m2)){
                foreach($m2[1] as $src){ if (strpos($src,'/pluginfile.php/')!==false) continue;
                    $total++; if ($total <= $offset || $shown >= $perpage) continue;
                    $thumb='<a href="'.s($src).'" target="_blank" rel="noopener noreferrer"><img referrerpolicy="no-referrer" src="'.s($src).'" style="max-width:160px;max-height:100px;object-fit:contain"></a>';
                    $matches[]=['where'=>$where,'act'=>$actlink,'render'=>$thumb]; $shown++;
                }
            }
            if (preg_match_all('/<img[^>]+src\s*=\s*"(data:image\/[a-zA-Z0-9.+-]+;base64,[^"<>]+)"/i',$html,$m1)){
                foreach($m1[1] as $src){
                    $total++; if ($total <= $offset || $shown >= $perpage) continue;
                    $matches[]=['where'=>$where,'act'=>$actlink,'render'=>'Data URI '.substr(sha1($src),0,12)]; $shown++;
                }
            }
        }
        $rs->close();
    }
    $pages = (int)ceil(max(1,$total) / max(1,$perpage));

    $basehtml = new moodle_url($base, ['tab'=>'html','columns'=>$columns,'perpage'=>$perpage]);
    ir_pager_form($basehtml, $page, $pages, ['tab'=>'html','columns'=>$columns,'perpage'=>$perpage]);
    $dlurl = new moodle_url($base, ['tab'=>'html','download'=>'csv','columns'=>$columns,'perpage'=>$perpage,'page'=>$page]);
    echo '<div class="controls"><a class="btnlink" href="'.s($dlurl->out(false)).'">Exporter en CSV</a></div>';

    echo '<table><tr><th>Où</th><th>CM ID (activité)</th><th>Aperçu / Lien</th></tr>';
    foreach ($matches as $m){
        echo '<tr><td>'.s($m['where']).'</td><td>'.$m['act'].'</td><td>'.$m['render'].'</td></tr>';
    }
    echo '</table>';

    ir_pager_form($basehtml, $page, $pages, ['tab'=>'html','columns'=>$columns,'perpage'=>$perpage]);
}
else { // dupes
    $countsql = "SELECT COUNT(1) FROM (SELECT contenthash FROM {files} WHERE filename<>'.' AND filesize>0 AND mimetype LIKE 'image/%' GROUP BY contenthash HAVING COUNT(*)>1) t";
    $total = $DB->count_records_sql($countsql);
    $pages = (int)ceil(max(1,$total) / max(1,$perpage));

    $basedup = new moodle_url($base, ['tab'=>'dupes','perpage'=>$perpage]);
    ir_pager_form($basedup, $page, $pages, ['tab'=>'dupes','perpage'=>$perpage]);
    $dlurl = new moodle_url($base, ['tab'=>'dupes','download'=>'csv','perpage'=>$perpage,'page'=>$page]);
    echo '<div class="controls"><a class="btnlink" href="'.s($dlurl->out(false)).'">Exporter en CSV</a></div>';

    $sql = "SELECT contenthash, COUNT(*) AS cnt, SUM(filesize) AS totalsize, MAX(timemodified) AS lastseen
              FROM {files}
             WHERE filename <> '.' AND filesize > 0 AND mimetype LIKE 'image/%'
          GROUP BY contenthash HAVING COUNT(*) > 1
          ORDER BY totalsize DESC";
    $groups = $DB->get_records_sql($sql, [], $page*$perpage, $perpage);

    echo '<table><tr><th>Aperçu</th><th>contenthash</th><th>Occur.</th><th>Taille totale</th><th>Exemples</th></tr>';
    foreach ($groups as $g) {
        $one = $DB->get_record_sql("SELECT f.id, f.filename, f.component, f.filearea,
                                           c.id AS courseid, cm.id AS cmid, m.name AS modname
                                      FROM {files} f
                                      JOIN {context} ctx ON ctx.id = f.contextid
                                 LEFT JOIN {course_modules} cm ON (ctx.contextlevel = ".CONTEXT_MODULE." AND cm.id = ctx.instanceid)
                                 LEFT JOIN {modules} m ON (m.id = cm.module)
                                 LEFT JOIN {course} c ON ((ctx.contextlevel = ".CONTEXT_COURSE." AND c.id = ctx.instanceid) OR (ctx.contextlevel = ".CONTEXT_MODULE." AND c.id = cm.course))
                                     WHERE f.contenthash = :h AND f.filename <> '.' AND f.filesize > 0
                                  ORDER BY f.timemodified DESC",
                                  ['h'=>$g->contenthash], IGNORE_MULTIPLE);
        $thumb = $one ? new moodle_url($base, ['action'=>'thumb','id'=>$one->id,'w'=>80,'h'=>80,'sesskey'=>sesskey()]) : null;
        $thumbhtml = $thumb ? '<img src="'.s($thumb->out(false)).'" style="max-width:80px;max-height:80px;object-fit:contain">' : '';
        $samples = $DB->get_records_sql("SELECT f.id, f.filename, f.component, f.filearea,
                                                c.id AS courseid, cm.id AS cmid, m.name AS modname
                                           FROM {files} f
                                           JOIN {context} ctx ON ctx.id = f.contextid
                                      LEFT JOIN {course_modules} cm ON (ctx.contextlevel = ".CONTEXT_MODULE." AND cm.id = ctx.instanceid)
                                      LEFT JOIN {modules} m ON (m.id = cm.module)
                                      LEFT JOIN {course} c ON ((ctx.contextlevel = ".CONTEXT_COURSE." AND c.id = ctx.instanceid) OR (ctx.contextlevel = ".CONTEXT_MODULE." AND c.id = cm.course))
                                          WHERE f.contenthash = :h AND f.filename <> '.' AND f.filesize > 0
                                       ORDER BY f.timemodified DESC",
                                       ['h'=>$g->contenthash], 0, 5);
        $samp = [];
        foreach ($samples as $s) {
            $orig = new moodle_url($base, ['action'=>'thumb','id'=>$s->id,'sesskey'=>sesskey()]);
            $coursebadge = !empty($s->courseid) ? ' — course <a href="'.s((new moodle_url('/course/view.php', ['id'=>$s->courseid]))->out(false)).'" target="_blank">'.(int)$s->courseid.'</a>' : '';
            $cmbadge = (!empty($s->cmid) && !empty($s->modname)) ? ' — cm <a href="'.s((new moodle_url('/mod/'.$s->modname.'/view.php', ['id'=>$s->cmid]))->out(false)).'" target="_blank">'.(int)$s->cmid.'</a>' : '';
            $samp[] = '<a href="'.s($orig->out(false)).'" target="_blank">'.s($s->filename).'</a> ('.s($s->component).'/'.s($s->filearea).')'.$coursebadge.$cmbadge;
        }
        echo '<tr>';
        echo '<td>'.$thumbhtml.'</td>';
        echo '<td>'.s(substr($g->contenthash,0,12)).'…</td>';
        echo '<td>'.(int)$g->cnt.'</td>';
        echo '<td>'.display_size($g->totalsize).'</td>';
        echo '<td>'.implode('<br>', $samp).'</td>';
        echo '</tr>';
    }
    echo '</table>';

    ir_pager_form($basedup, $page, $pages, ['tab'=>'dupes','perpage'=>$perpage]);
}

echo '</body></html>';

// ---------- CSV Exports ----------
if ($download==='csv'){
    if ($tab==='files') ir_files_csv($componentf,$fileareaf,$search);
    else if ($tab==='html') ir_html_csv($columns);
    else ir_dupes_csv();
    exit;
}

function ir_files_csv($componentf,$fileareaf,$search){
    global $DB,$CFG;
    require_once($CFG->libdir.'/csvlib.class.php');
    $where = ir_where($params,$componentf,$fileareaf,$search);
    $sql = "SELECT f.id,f.contextid,f.component,f.filearea,f.itemid,f.filename,f.mimetype,f.filesize,f.timecreated,f.timemodified,
                   ctx.contextlevel, ctx.instanceid,
                   c.id AS courseid, c.fullname AS coursename,
                   cm.id AS cmid, cm.instance AS instanceid, m.name AS modname
              FROM {files} f
              JOIN {context} ctx ON ctx.id = f.contextid
         LEFT JOIN {course_modules} cm ON (ctx.contextlevel = ".CONTEXT_MODULE." AND cm.id = ctx.instanceid)
         LEFT JOIN {modules} m ON (m.id = cm.module)
         LEFT JOIN {course} c ON ((ctx.contextlevel = ".CONTEXT_COURSE." AND c.id = ctx.instanceid) OR (ctx.contextlevel = ".CONTEXT_MODULE." AND c.id = cm.course))
              $where
          ORDER BY f.timemodified DESC";
    $rs = $DB->get_recordset_sql($sql, $params);
    $csv = new csv_export_writer(); $csv->set_filename('imagereport_files');
    $csv->add_data(['id','contextid','component','filearea','itemid','filename','mimetype','filesize','timecreated','timemodified','courseid','coursename','cmid','modname','instanceid']);
    foreach($rs as $r){
        $csv->add_data([$r->id,$r->contextid,$r->component,$r->filearea,$r->itemid,$r->filename,$r->mimetype,$r->filesize,userdate($r->timecreated),userdate($r->timemodified),(string)$r->courseid,(string)$r->coursename,(string)$r->cmid,(string)$r->modname,(string)$r->instanceid]);
    }
    $rs->close(); $csv->download_file();
}

function ir_html_csv($columns){
    global $DB,$CFG;
    require_once($CFG->libdir.'/csvlib.class.php');
    $cols = array_filter(array_map('trim', explode(',', (string)$columns)));
    $csv = new csv_export_writer(); $csv->set_filename('imagereport_htmlscan');
    $csv->add_data(['table','rowid','courseid','cmid','modname','kind','src']);
    foreach ($cols as $tc){
        if (strpos($tc,'.')===false) continue;
        list($table,$col)=explode('.',$tc,2);
        $table=preg_replace('/[^a-z0-9_]/','',$table);
        $col  =preg_replace('/[^a-z0-9_]/','',$col);
        if (!$DB->get_manager()->table_exists($table)) continue;
        $hascourse = array_key_exists('course', $DB->get_columns($table));
        $select = $hascourse ? "id, $col AS html, course AS courseid" : "id, $col AS html, NULL AS courseid";
        $sql="SELECT $select FROM {".$table."} WHERE $col LIKE :n1 OR $col LIKE :n2";
        $rs=$DB->get_recordset_sql($sql,['n1'=>'%<img%','n2'=>'%data:image%']);
        foreach($rs as $row){
            $html=(string)$row->html; if ($html==='') continue;
            $resolved = ir_resolve_cmid_for($table, $row->id);
            $cmid = $resolved['cmid']; $modname = $resolved['modname'];
            if (preg_match_all('/<img[^>]+src\s*=\s*"(https?:[^"\s<>]+)"/i',$html,$m2)){
                foreach($m2[1] as $u){ if (strpos($u,'/pluginfile.php/')!==false) continue; $csv->add_data([$table,$row->id,(string)$row->courseid,(string)$cmid,(string)$modname,'external',$u]); }
            }
            if (preg_match_all('/<img[^>]+src\s*=\s*"(data:image\/[a-zA-Z0-9.+-]+;base64,[^"<>]+)"/i',$html,$m1)){
                foreach($m1[1] as $d){ $csv->add_data([$table,$row->id,(string)$row->courseid,(string)$cmid,(string)$modname,'data',substr(sha1($d),0,12)]); }
            }
        }
        $rs->close();
    }
    $csv->download_file();
}

function ir_dupes_csv(){
    global $DB,$CFG; require_once($CFG->libdir.'/csvlib.class.php');
    $csv=new csv_export_writer(); $csv->set_filename('imagereport_duplicates');
    $csv->add_data(['contenthash','count','totalsize','lastseen','samples']);
    $sql = "SELECT contenthash, COUNT(*) AS cnt, SUM(filesize) AS totalsize, MAX(timemodified) AS lastseen
              FROM {files}
             WHERE filename <> '.' AND filesize > 0 AND mimetype LIKE 'image/%'
          GROUP BY contenthash HAVING COUNT(*) > 1
          ORDER BY totalsize DESC";
    $rs = $DB->get_recordset_sql($sql);
    foreach ($rs as $g){
        $samples = $DB->get_records_sql("SELECT f.filename, f.component, f.filearea, c.id AS courseid, cm.id AS cmid, m.name AS modname
                                           FROM {files} f
                                           JOIN {context} ctx ON ctx.id = f.contextid
                                      LEFT JOIN {course_modules} cm ON (ctx.contextlevel = ".CONTEXT_MODULE." AND cm.id = ctx.instanceid)
                                      LEFT JOIN {modules} m ON (m.id = cm.module)
                                      LEFT JOIN {course} c ON ((ctx.contextlevel = ".CONTEXT_COURSE." AND c.id = ctx.instanceid) OR (ctx.contextlevel = ".CONTEXT_MODULE." AND c.id = cm.course))
                                          WHERE f.contenthash = :h AND f.filename <> '.' AND f.filesize > 0
                                       ORDER BY f.timemodified DESC",
                                       ['h'=>$g->contenthash], 0, 5);
        $s = [];
        foreach ($samples as $row) {
            $s[] = $row->filename.' ('.$row->component.'/'.$row->filearea.')'
                  .($row->courseid ? ' [course '.$row->courseid.']' : '')
                  .($row->cmid && $row->modname ? ' [cm '.$row->cmid.' ('.$row->modname.')]':'')
                  ;
        }
        $csv->add_data([$g->contenthash,$g->cnt,$g->totalsize,userdate($g->lastseen), implode(' | ', $s)]);
    }
    $rs->close(); $csv->download_file();
}
